package ppbot;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;


class GeneticAlgorithm implements Serializable {
    private double bestFitness;

    private void markElite() {        
        for(GeneticIndividual indi : pop)
            indi.setElite(false);
        bestFitness = bestGeneticIndividual.getFitness();
        bestGeneticIndividual.setElite(true);
    }

    private class Population extends ArrayList<GeneticIndividual> implements Serializable {
        
        public Population(int size) 
        {
            super(size);            
        }
        
        public GeneticIndividual selectByTournament(int rounds)
        {
            int champion = (int)(size() * Math.random());
            for(int i=1; i<rounds; i++)
            {
                int challenger = (int)(size() * Math.random());
                if (get(challenger).getFitness() > get(champion).getFitness())
                    champion = challenger;                
            }
            return get(champion);                
        }
        
        public void initializeRandomly()
        {
            for (GeneticIndividual i : pop) {
                i.initializeRandomly();
            }            
        }
        
        public GeneticIndividual getBest()
        {
            GeneticIndividual best = null;
            for(GeneticIndividual indi : this)
                if ((best == null) || (indi.getFitness() > best.getFitness())) best = indi;
            return best;
        }
        
    }

    public Population pop;
    
    public double mutRate;
    
    public double crossRate;
    
    private int nextIndex = 0;
    
    private GeneticIndividual bestGeneticIndividual = null;
    
    public GeneticAlgorithm(int popSize, GeneticIndividual first) 
    {
        pop = new Population(popSize);
        for(int i=0; i<popSize; i++)
            pop.add(first.clone());
        
    }
    
    public void initializePopulationRandomly()
    {
        pop.initializeRandomly();
    }
        
    private void mutate()
    {
        for (GeneticIndividual i : pop) {
            i.mutate(mutRate);
        }
    }
    
    private void select()
    {
        Population newPop = new Population(pop.size());
        for(GeneticIndividual indi : pop)
            if (indi.isElite()) {
                newPop.add(indi.clone());
                break;
            }
        
        while (newPop.size() < pop.size())    
        {
            GeneticIndividual selected = pop.selectByTournament(2);
            newPop.add(selected.clone());
        }
        pop = newPop;
    }
    
    private void crossover()
    {
        GeneticIndividual parent = null;
        for (GeneticIndividual indi : pop) {
            if (Math.random() < crossRate)
            {
                if (parent == null) parent = indi;
                else {
                    parent.crossover(indi);
                    parent = null;
                }
            }                    
        }
        
    }
    
    public void step()
    {
        // expects the fitness is already evaluated
        
        bestGeneticIndividual = pop.getBest();
        
        markElite();
        
        select();
        
        mutate();
        
        crossover();
        
        nextIndex = 0;
        
    }                 
    
    public GeneticIndividual getNext()
    {
        if (nextIndex < pop.size())
            return pop.get(nextIndex++);
        else return null;
    }
    
    public String populationToString()
    {
        StringBuilder str = new StringBuilder();
        for(GeneticIndividual indi : pop) {
            str.append(indi.toString());
            str.append("\n");
        }
        return str.toString();
    }
    
    public double getPopulationAverageFitness()
    {
        double sum = 0;
        for(GeneticIndividual indi : pop) 
            sum+=indi.getFitness();
        return sum / pop.size();            
    }
    
    public double getPopulationBestFitness()
    {
        return bestGeneticIndividual.getFitness();
    }    
    
    public GeneticIndividual getBestIndividual()
    {
        return bestGeneticIndividual;
    }
    
    public static void main(String[] args)
    {
        int[] layers = {1,2};
        GeneticAlgorithm GA = new GeneticAlgorithm(5, new NeuralNetwork(layers));
        
        GA.initializePopulationRandomly();
        
        for(int i=0; i<100; i++)
        {
            NeuralNetwork net = (NeuralNetwork) GA.getNext();
            
            if (net == null) 
            {
                GA.step();
                net = (NeuralNetwork) GA.getNext();                        
                System.out.println("--------------");
            }
            System.out.println(net.hashCode());
            net.geneticFitness = Math.random();            
        }        
    }
    
    public void serializePopulation(ObjectOutputStream oos) throws IOException
    {
        for (GeneticIndividual indi : pop) 
            indi.serialize(oos);
    }
    
    public void deserializePopulation(ObjectInputStream ois) throws IOException, ClassNotFoundException 
    {
        for (GeneticIndividual indi : pop)
            indi.deserialize(ois);
    }
    
}
